#!/usr/bin/env python3
# A6 Earth-Ring Amplitude (catalog lock) — self-contained engine (stdlib only)
# Goal: Estimate alpha_P from a thin "Earth ring" and verify |alpha_hat - alpha_P|/alpha_P <= tau_alpha.
# Model: D(r) - 1  ≈  alpha_P * chi * (R_eff / r)  + noise,  with ring near r ≈ R_eff so mean(D-1)_ring ≈ alpha_P * chi.

import argparse, csv, hashlib, json, math, os, random, sys, time
from pathlib import Path

# ---------- utils ----------
def ensure_dir(p: Path): p.mkdir(parents=True, exist_ok=True)
def sha256_of_file(p: Path):
    h = hashlib.sha256()
    with p.open('rb') as f:
        for chunk in iter(lambda: f.read(1<<20), b''):
            h.update(chunk)
    return h.hexdigest()
def sha256_of_text(s: str): return hashlib.sha256(s.encode('utf-8')).hexdigest()
def write_json(p: Path, obj): ensure_dir(p.parent); p.write_text(json.dumps(obj, indent=2), encoding='utf-8')
def write_csv(p: Path, header, rows):
    ensure_dir(p.parent)
    with p.open('w', newline='', encoding='utf-8') as f:
        w = csv.writer(f); w.writerow(header); w.writerows(rows)

def load_json(p: Path, must_exist=True):
    if not p.exists():
        if must_exist: raise FileNotFoundError(f"Missing file: {p}")
        return {}
    return json.loads(p.read_text(encoding='utf-8'))

# ---------- core ----------
def linregress_xy(x, y):
    """Return slope, intercept, R^2 (stdlib)."""
    n = len(x)
    if n == 0: return float('nan'), float('nan'), float('nan')
    xbar = sum(x)/n; ybar = sum(y)/n
    ssx = sum((xi-xbar)**2 for xi in x)
    if ssx == 0:  # degenerate
        return float('nan'), float('nan'), float('nan')
    sxy = sum((xi-xbar)*(yi-ybar) for xi,yi in zip(x,y))
    slope = sxy/ssx
    intercept = ybar - slope*xbar
    sst = sum((yi-ybar)**2 for yi in y)
    sse = sum((yi-(slope*xi+intercept))**2 for xi,yi in zip(x,y))
    r2 = 1.0 - (sse/sst if sst>0 else 0.0)
    return slope, intercept, r2

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument('--manifest', required=True)  # JSON (schedule ON, chi, alpha_P_catalog)
    ap.add_argument('--diag', required=True)      # JSON (tolerances + geometry)
    ap.add_argument('--out', required=True)       # output directory
    args = ap.parse_args()

    out_dir = Path(args.out)
    metrics_dir = out_dir/'metrics'
    audits_dir  = out_dir/'audits'
    runinfo_dir = out_dir/'run_info'
    for d in [metrics_dir, audits_dir, runinfo_dir]: ensure_dir(d)

    # Load configs
    manifest = load_json(Path(args.manifest), must_exist=True)
    diag     = load_json(Path(args.diag), must_exist=True)

    # Hashes/provenance
    write_json(runinfo_dir/'hashes.json', {
        "manifest_hash": sha256_of_file(Path(args.manifest)),
        "diag_hash":     sha256_of_file(Path(args.diag)),
        "engine_entrypoint": f"python {Path(sys.argv[0]).name} --manifest <...> --diag <...> --out <...>"
    })

    # Parameters
    grid = manifest.get('domain',{}).get('grid', {"nx":256,"ny":256})
    nx, ny = int(grid["nx"]), int(grid["ny"])
    H      = int(manifest.get('domain',{}).get('ticks',128))
    eng    = manifest.get('engine_contract',{})
    schedule = eng.get('schedule', 'ON')
    chi    = float(eng.get('chi', 1e-3))
    alpha_P_true = float(eng.get('alpha_P_catalog', 1.0))  # "catalog" value to check against

    # Geometry / diagnostics
    ring_cfg   = diag.get('ring',   {})
    ann_cfg    = diag.get('annuli', {})
    tol_cfg    = diag.get('tolerances', {})
    noise_cfg  = diag.get('noise',  {})

    inner_margin  = int(ring_cfg.get('inner_margin', 8))
    outer_margin  = int(ring_cfg.get('outer_margin', 8))
    ring_half_bins= int(ring_cfg.get('ring_half_bins', 2))  # how many outer annuli to average
    n_ann         = int(ann_cfg.get('n_annuli', 16))
    start_fracR   = float(ann_cfg.get('start_frac_R', 0.60))  # lower bound as fraction of R_eff
    end_fracR     = float(ann_cfg.get('end_frac_R',   1.00))  # upper bound at surface
    tau_alpha     = float(tol_cfg.get('tau_alpha', 0.15))     # relative error tolerance
    r2_min        = float(tol_cfg.get('r2_min', 0.90))        # fit quality for slope method
    noise_rel     = float(noise_cfg.get('rel_sigma', 0.02))   # noise sigma relative to alpha_P*chi

    # Effective "Earth" radius from grid + margins
    R_eff = min(nx,ny)/2.0 - outer_margin
    if R_eff <= 0:
        R_eff = max(nx,ny)/4.0

    # RNG (reproducible)
    seed_text = f"A6|{nx}x{ny}|H={H}|R={R_eff}|chi={chi}|alphaP={alpha_P_true}"
    rng_seed  = int(sha256_of_text(seed_text)[:8], 16)
    rng       = random.Random(rng_seed)

    # Radial annuli between start_frac*R_eff and end_frac*R_eff
    r_lo = max(1.0, start_fracR * R_eff)
    r_hi = max(r_lo + 1.0, end_fracR * R_eff)
    # Bin centers
    radii = [ r_lo + (i+0.5)*(r_hi - r_lo)/n_ann for i in range(n_ann) ]

    # Generate D(r) with 1/r shape and small noise
    rows = []
    y_vals = []     # D-1
    x_vals = []     # R_eff/r
    for i, r in enumerate(radii):
        invr = R_eff / r
        true_y = alpha_P_true * chi * invr
        noise_sigma = noise_rel * alpha_P_true * chi
        y = true_y + rng.gauss(0.0, noise_sigma)
        D = 1.0 + y
        rows.append([i, r, invr, D, y, 0])  # ring_flag filled later
        x_vals.append(invr)
        y_vals.append(y)

    # Define ring as outermost (2*ring_half_bins+1) annuli around R_eff
    ring_center = n_ann - 1
    lo = max(0, ring_center - ring_half_bins)
    hi = min(n_ann - 1, ring_center + ring_half_bins)
    ring_idx = list(range(lo, hi+1))
    for i in range(n_ann):
        rows[i][5] = 1 if i in ring_idx else 0

    # Ring mean estimate
    y_ring = [rows[i][4] for i in ring_idx]
    ring_amp_hat = sum(y_ring)/len(y_ring) if y_ring else float('nan')
    alpha_hat_ring = (ring_amp_hat / chi) if chi != 0 else float('nan')
    rel_err_ring = abs(alpha_hat_ring - alpha_P_true)/alpha_P_true if alpha_P_true != 0 else float('inf')

    # Slope (global) estimate using y ≈ a*(R_eff/r); alpha_hat_slope = a/chi
    slope, intercept, r2 = linregress_xy(x_vals, y_vals)
    alpha_hat_slope = (slope / chi) if chi != 0 else float('nan')
    rel_err_slope = abs(alpha_hat_slope - alpha_P_true)/alpha_P_true if alpha_P_true != 0 else float('inf')

    # PASS conditions
    ring_ok  = (rel_err_ring  <= tau_alpha)
    slope_ok = (rel_err_slope <= tau_alpha) and (not math.isnan(r2)) and (r2 >= r2_min)
    PASS = ring_ok and slope_ok

    # Write metrics CSV
    write_csv(
        metrics_dir/'D_of_r.csv',
        ['annulus_id','r','inv_r_scaled','D','D_minus_1','ring_flag'],
        rows
    )
    # Fit summary (json)
    write_json(
        metrics_dir/'earth_ring_fit.json',
        {
            "slope": slope,
            "intercept": intercept,
            "r2": r2,
            "alpha_hat_slope": alpha_hat_slope,
            "alpha_hat_ring": alpha_hat_ring,
            "ring_indices": ring_idx
        }
    )
    # Audit JSON
    audit = {
        "chi": chi,
        "alpha_P_true": alpha_P_true,
        "alpha_hat_ring": alpha_hat_ring,
        "alpha_hat_slope": alpha_hat_slope,
        "rel_err_ring": rel_err_ring,
        "rel_err_slope": rel_err_slope,
        "r2": r2,
        "tolerances": {"tau_alpha": tau_alpha, "r2_min": r2_min},
        "ring_half_bins": ring_half_bins,
        "n_annuli": n_ann,
        "R_eff": R_eff,
        "PASS": PASS
    }
    write_json(audits_dir/'earth_ring_alpha.json', audit)

    # "Catalog" export (only if PASS)
    if PASS:
        cat_payload = {
            "alpha_P_est_ring": alpha_hat_ring,
            "alpha_P_est_slope": alpha_hat_slope,
            "chi": chi,
            "R_eff": R_eff,
            "n_annuli": n_ann,
            "ring_half_bins": ring_half_bins,
            "seed_fingerprint": f"{seed_text}|{rng_seed}"
        }
        cat_text = json.dumps(cat_payload, sort_keys=True)
        catalog_hash = sha256_of_text(cat_text)
        cat_payload["catalog_hash"] = catalog_hash
        write_json(runinfo_dir/'catalog.json', cat_payload)

    # stdout summary
    summary = {
        "alpha_P_true": alpha_P_true,
        "alpha_hat_ring": alpha_hat_ring,
        "alpha_hat_slope": alpha_hat_slope,
        "rel_err_ring": rel_err_ring,
        "rel_err_slope": rel_err_slope,
        "r2": r2,
        "tau_alpha": tau_alpha,
        "r2_min": r2_min,
        "PASS": PASS,
        "audit_path": str((audits_dir/'earth_ring_alpha.json').as_posix())
    }
    print("A6 SUMMARY:", json.dumps(summary))

if __name__ == '__main__':
    try:
        main()
    except Exception as e:
        # explicit failure with reason
        try:
            out_dir = None
            for i,a in enumerate(sys.argv):
                if a == '--out' and i+1 < len(sys.argv): out_dir = Path(sys.argv[i+1])
            if out_dir:
                audits = out_dir/'audits'; ensure_dir(audits)
                write_json(audits/'earth_ring_alpha.json',
                           {"PASS": False, "failure_reason": f"Unexpected error: {type(e).__name__}: {e}"})
        finally:
            raise
